/*
*
* Modified from sample code provided by Joey de Vries at https://learnopengl.com/,
* as permitted under CC BY-NC 4.0 license (https://creativecommons.org/licenses/by-nc/4.0/legalcode)
*
* Copyright (c) Joey de Vries (https://learnopengl.com/, https://twitter.com/JoeyDeVriez)
*/

#include <iostream>
#include <fstream>
#include <string>
#include <sstream>

#define GLEW_STATIC
#define GLM_SWIZZLE
#define GLM_FORCE_RADIANS
#include <time.h>
#include <GL/glew.h>
#include <GLFW/glfw3.h>
#include <GL/glext.h>
#include <math.h>
#include <vector>
//#include <random>

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/matrix_access.hpp>
#include <glm/gtx/norm.hpp>
#include <glm/gtc/type_ptr.hpp>
#include <glm/ext.hpp>

#include <SOIL/SOIL.h>
#include "main.h"
#include "shader.h"
//#include "Model.h"
#include "ModelLite.h"
#include "Mesh.h"
#include "envMaps.h"


glm::mat4 getInverseRotation(glm::mat4 modelMatrix) {
    glm::vec4 col = glm::column(modelMatrix, 0);
    float sf = 1.0 / sqrt(col[0]*col[0] + col[1]*col[1] + col[2]*col[2] + col[3]*col[3]);
    return glm::inverse(glm::scale(modelMatrix, glm::vec3(sf, sf, sf)));
}

void rotateEnv(float *env, float *envRot, float envAngle) {
    envRot[0] = env[0];

    // Band 1
    for (int l = 1; l < 9; l++) {
        for (int m = -l; m <= l; m++) {

            int ind = l*(l+1)+m;
            if (m == 0) {
                envRot[ind] = env[ind];
            }
            else {
                int opInd = l*(l+1)-m;
                float alpha = abs(m) * envAngle;
                int sign = m > 0 ? 1 : -1;
                envRot[ind] = cos(alpha) * env[ind] + sign * sin(alpha) * env[opInd];
            }
        }
    }
}

void setupLegendreTexture(GLuint texName, GLfloat data[][4], int legendre_res) {
    glBindTexture(GL_TEXTURE_1D, texName);
    glTexImage1D(GL_TEXTURE_1D, 0, GL_RGBA32F, legendre_res, 0, GL_RGBA, GL_FLOAT, data);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_MIRRORED_REPEAT);
}


void loadEnvMap(GLuint envMap, GLchar* path) {
    int envWidth, envHeight;
    unsigned char* envImg = SOIL_load_image(path, &envWidth, &envHeight, 0, SOIL_LOAD_RGBA);

    glBindTexture(GL_TEXTURE_2D, envMap);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, envWidth, envHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, envImg);
    glGenerateMipmap(GL_TEXTURE_2D);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    glBindTexture(GL_TEXTURE_2D, 0);

    SOIL_free_image_data(envImg);

}

int main() {

    glfwInit();
    
    glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
    glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
    glfwWindowHint(GLFW_RESIZABLE, GL_FALSE);
    //glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);       // For Mac OS.
    
    // This function returns a GLFWwindow object
    GLFWwindow* window = glfwCreateWindow(1000, 1000, "OpenGL Demo", NULL, NULL);
    if (window == NULL) {
        std::cout << "Failed to create GLFW window" << std::endl;
        glfwTerminate();
    }
    glfwMakeContextCurrent(window);
    glfwSetKeyCallback(window, key_callback);
    
    glewExperimental = GL_TRUE;
    if (glewInit() != GLEW_OK) {
        std::cout << "Failed to initialize GLEW" << std::endl;
    }

    int width, height;
    glfwGetFramebufferSize(window, &width, &height);
    glViewport(0, 0, width, height);

    Model scene("dragon_floats/dragon_lower");

    int result;
    glGetIntegerv(GL_MAX_TEXTURE_BUFFER_SIZE_ARB, &result);
    cout << "max texture buffer size: " << result << endl;    
    
    // Read in SH coefficients
    int numVerts = scene.numVertices;
    GLfloat *TransferCoeffs = new float [numVerts * 84];

    char *byteData = &(scene.coeffData[0]);
    float *mCoeffs = reinterpret_cast<float *> (byteData);
    
    for (int i = 0; i < numVerts; i++) {
        int startInd = i * 84;
        for (int j = 0; j < 81; j++) {
            TransferCoeffs[startInd] = mCoeffs[i*81 + j];
            startInd++;
        }
    }    

    GLuint TexBuffer;
    glGenBuffers(1, &TexBuffer);
    glBindBuffer(GL_TEXTURE_BUFFER_ARB, TexBuffer);
    glBufferData(GL_TEXTURE_BUFFER_ARB, numVerts * 84 * sizeof(GLfloat), TransferCoeffs, GL_STATIC_DRAW);
    delete [] TransferCoeffs;
    glTexBufferARB(GL_TEXTURE_BUFFER_ARB, GL_RGBA32F, TexBuffer);

    model = glm::mat4(1.0);
    model = glm::translate(model, glm::vec3(-0.0f, 0.0, -0.05));
    lightTransform = glm::translate(lightTransform, glm::vec3(-0.0f, 0.0, -0.05));

    memcpy(glm::value_ptr(model), model_init, sizeof(model_init));
    memcpy(glm::value_ptr(lightTransform), light_init, sizeof(light_init));

    
    glm::mat4 view;
    view = glm::lookAt(cameraPos, cameraTarget, cameraUp);
    glm::mat4 projection;
    projection = glm::perspective(glm::radians(45.0f), float(width)/float(height), 0.1f, 250.0f);
    
    GLuint lightVAO;
    GLuint lightVBO;
    GLuint lightEBO;
    
    glGenBuffers(1, &lightVBO);
    glGenBuffers(1, &lightEBO);
    glGenVertexArrays(1, &lightVAO);
    
    glBindVertexArray(lightVAO);
    {
        glBindBuffer(GL_ARRAY_BUFFER, lightVBO);
        glBufferData(GL_ARRAY_BUFFER, sizeof(lightverts), lightverts, GL_STATIC_DRAW);
        glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 3 * sizeof(GLfloat), (GLvoid*) 0);
        glEnableVertexAttribArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, 0);   // Unbind
        
        // Bind the EBO
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, lightEBO);
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(lightinds), lightinds, GL_STATIC_DRAW);
    }
    glBindVertexArray(0);

    std::vector<GLchar*> vshader2;
    std::vector<GLchar*> fshader2;

    vshader2.push_back("opt/ours_band8.glsl");
    fshader2.push_back("opt/frag.glsl");

    Shader shadepass1(vshader2, fshader2);

    glm::vec4 lightvertices[16] = {
       
        glm::vec4(-0.05, -0.08, 0.2, 1),
        glm::vec4(0.01, -0.08, 0.2, 1),
        glm::vec4(0.01, 0.0, 0.2, 1),
        glm::vec4(-0.05, 0.0, 0.2, 1)

    };

    glm::vec3 lights[16];

    int legendre_res = 10000;
    GLfloat legendre_2345[legendre_res][4];
    GLfloat legendre_6789[legendre_res][4];

    // Our iterative method requires values of Legendre polynomials
    // Here we store them in textures so they can be accessed in the shader in ~constant time
    // Also take advantage of symmetry properties of polynomials with texture mirroring

    for (int i = 0; i < legendre_res; i++) {
        float X[10];
        X[1] = float(i)/legendre_res;
        for (int j = 2; j < 10; j++) {
            X[j] = X[j-1] * X[1];
        }
        
        legendre_2345[i][0] = (0.5 * (3 * X[2] -1));
        legendre_2345[i][1] = (0.5 * (5 * X[3] - 3*X[1]));
        legendre_2345[i][2] = ((35 * X[4] - 30*X[2]+ 3)/ 8.0);
        legendre_2345[i][3] = ((63 * X[5] - 70*X[3] + 15*X[1])/8.0);

        legendre_6789[i][0] = ((231*X[6] - 315*X[4] + 105*X[2] - 5)/16.0);
        legendre_6789[i][1] = ((429*X[7] - 693*X[5] + 315*X[3] - 35*X[1])/16.0);
        legendre_6789[i][2] = ((6435*X[8] - 12012*X[6] + 6930*X[4] - 1260*X[2] + 35)/128.0);
        legendre_6789[i][3] = ((12155*X[9] - 25740*X[7] + 18018*X[5] - 4620*X[3] + 315*X[1])/128.0); 
       
    }

    GLuint LUT_L2345, LUT_L6789;

    glGenTextures(1, &LUT_L2345);
    setupLegendreTexture(LUT_L2345, legendre_2345, legendre_res);
    glGenTextures(1, &LUT_L6789);
    setupLegendreTexture(LUT_L6789, legendre_6789, legendre_res);
    

    glActiveTexture(GL_TEXTURE1);
    glBindTexture(GL_TEXTURE_BUFFER_ARB, TexBuffer);


    // Envmap for fullscreen quad
    GLuint envMap;
    glGenTextures(1, &envMap);
    loadEnvMap(envMap, "envs/grace.png");


    // Fullscreen quad
    GLuint quadVAO;
    glGenVertexArrays(1, &quadVAO);
    glBindVertexArray(quadVAO);

    const GLfloat g_quad_vertex_buffer_data[] = { 
        -1.0f, 1.0f, -1.0f, -1.0f, 1.0f, 1.0f, 1.0f, -1.0f };

    GLuint quad_vertexbuffer;
    glGenBuffers(1, &quad_vertexbuffer);
    glBindBuffer(GL_ARRAY_BUFFER, quad_vertexbuffer);
    glBufferData(GL_ARRAY_BUFFER, sizeof(g_quad_vertex_buffer_data), g_quad_vertex_buffer_data, GL_STATIC_DRAW);
    glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, NULL);
    glEnableVertexAttribArray(0);
    glBindVertexArray(0);


    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);
    glEnable(GL_BLEND);
    
    while (!glfwWindowShouldClose(window)) {
        glfwPollEvents();
        glClearColor(0.3f, 0.3f, 0.3f, 1.0f);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        shadepass1.Use();
        
        glUniform3fv(glGetUniformLocation(shadepass1.Program, "eyePos"), 1,
                     glm::value_ptr(cameraPos));
        
        /////////// Model, view, projection matrices
        glUniformMatrix4fv(glGetUniformLocation(shadepass1.Program, "view"),
                           1, GL_FALSE, glm::value_ptr(view));
        glUniformMatrix4fv(glGetUniformLocation(shadepass1.Program, "projection"),
                           1, GL_FALSE, glm::value_ptr(projection));

        glUniform1i(glGetUniformLocation(shadepass1.Program, "isLight"), 0);     // Reset isLight
        glUniformMatrix4fv(glGetUniformLocation(shadepass1.Program, "model"), 1, // Reset model matrix
                           GL_FALSE, glm::value_ptr(identity));

        // Render fullscreen quad to show env map texture
        
        glDepthMask(GL_FALSE);
        glUniform1i(glGetUniformLocation(shadepass1.Program, "IBL"), IBL);

        
        glActiveTexture(GL_TEXTURE4);  
        glBindTexture(GL_TEXTURE_2D, envMap);

        glUniform1i(glGetUniformLocation(shadepass1.Program, "envMap"), 4);

        glUniform1i(glGetUniformLocation(shadepass1.Program, "isBackground"), 1);
        glBlendFunc(GL_ONE, GL_ZERO);
        glBindVertexArray(quadVAO);
        glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        glBindVertexArray(0);
        glUniform1i(glGetUniformLocation(shadepass1.Program, "isBackground"), 0);
        glDepthMask(GL_TRUE);
        
        ///////////
        
        // Environment lighting coefficients    
        rotateEnv(envRed, envRedRot, envAngle);
        rotateEnv(envGreen, envGreenRot, envAngle);
        rotateEnv(envBlue, envBlueRot, envAngle);
            
        glUniform1fv(glGetUniformLocation(shadepass1.Program, "envRed"), 81, (const GLfloat*)envRedRot);
        glUniform1fv(glGetUniformLocation(shadepass1.Program, "envGreen"), 81, (const GLfloat*)envGreenRot);
        glUniform1fv(glGetUniformLocation(shadepass1.Program, "envBlue"), 81, (const GLfloat*)envBlueRot);
        
        
        ///////////
        glActiveTexture(GL_TEXTURE2);
        glUniform1i(glGetUniformLocation(shadepass1.Program, "LUT_L2345"), 2);
        glBindTexture(GL_TEXTURE_1D, LUT_L2345);

        glActiveTexture(GL_TEXTURE3);
        glUniform1i(glGetUniformLocation(shadepass1.Program, "LUT_L6789"), 3);
        glBindTexture(GL_TEXTURE_1D, LUT_L6789);
        
        glActiveTexture(GL_TEXTURE1);
        glUniform1i(glGetUniformLocation(shadepass1.Program, "LUT_coeffs"), 1);
        glBindTexture(GL_TEXTURE_BUFFER_ARB, TexBuffer);

        
        // Make area light vertices are available to all fragments to calculate lighting
        // Because of how visibility is precomputed (in world space) we apply the model's inverse
        // rotation to the area light to avoid having to rotate the visibility coefficients
        glm::mat4 invModelRot = getInverseRotation(model);

        for (int i = 0; i < 4; i++) {
            glm::vec4 _vert = lightTransform * (glm::vec4(lightScale, lightScale, 1, 1) * lightvertices[i]);
            _vert = invModelRot * _vert;
            lights[i] = _vert.xyz() / _vert.w;
        }
        glUniform3fv(glGetUniformLocation(shadepass1.Program, "lightSources"), 4, (const GLfloat*)lights);
        glUniformMatrix4fv(glGetUniformLocation(shadepass1.Program, "model"), 1,
                           GL_FALSE, glm::value_ptr(model));


        /////////// Draw scene
        glUniform1i(glGetUniformLocation(shadepass1.Program, "showEnv"), showEnv);
        glUniform1i(glGetUniformLocation(shadepass1.Program, "showArea"), showArea);
        glUniform1i(glGetUniformLocation(shadepass1.Program, "lightID"), 1);

        glUniform1f(glGetUniformLocation(shadepass1.Program, "lightScale"), lightScale);


        glBlendFunc(GL_ONE, GL_ZERO);
        scene.Draw(shadepass1, 1);


        glUniformMatrix4fv(glGetUniformLocation(shadepass1.Program, "model"),
                           1, GL_FALSE, glm::value_ptr(lightTransform));           // Model matrix for area light itself
        
        
        if (showLight && showArea) {
            glUniform1i(glGetUniformLocation(shadepass1.Program, "isLight"), 1);
            glBlendFunc(GL_ONE, GL_ZERO);
            glBindVertexArray(lightVAO);
            glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, 0);                    // Draw area light polygon
            glBindVertexArray(0);
        }

        glfwSwapBuffers(window);

    }

    glfwTerminate();
    return 0;
}

void key_callback(GLFWwindow* window, int key, int scancode, int action, int mode) {
    if (key == GLFW_KEY_ESCAPE && action == GLFW_PRESS)
        glfwSetWindowShouldClose(window, GL_TRUE);   

    if (key == GLFW_KEY_A) {
        model = glm::rotate(model, glm::radians(2.0f), glm::vec3(0, 0, 1));
        envAngle += glm::radians(2.0f);
    }
    if (key == GLFW_KEY_D) {
        model = glm::rotate(model, glm::radians(-2.0f), glm::vec3(0, 0, 1));
        envAngle -= glm::radians(2.0f);
    }

    if (key == GLFW_KEY_S) {
        model = glm::rotate(model, glm::radians(2.0f), glm::vec3(0, 1, 0));
    }
    if (key == GLFW_KEY_W) {
        model = glm::rotate(model, glm::radians(-2.0f), glm::vec3(0, 1, 0));
    }


    if (key == GLFW_KEY_LEFT) {
        lightTransform = glm::rotate(lightTransform, glm::radians(2.0f), up);
    }
    if (key == GLFW_KEY_RIGHT) {
        lightTransform = glm::rotate(lightTransform, glm::radians(-2.0f), up);
    }
    if (key == GLFW_KEY_UP) {
        lightTransform = glm::rotate(lightTransform, glm::radians(3.0f), glm::vec3(0,1,0));
    }
    if (key == GLFW_KEY_DOWN) {
        lightTransform = glm::rotate(lightTransform, glm::radians(-3.0f), glm::vec3(0,1,0));
    }


    if (key == GLFW_KEY_J && action == GLFW_RELEASE) {
        showEnv = 1 - showEnv;
    }
    if (key == GLFW_KEY_N && action == GLFW_RELEASE) {
        showArea = 1 - showArea;
    }
    
}

